# Autor: Tomáš Nedvěd
# Vedoucí práce: RNDr. Jakub Miřijovský, Ph.D.
# Univerzita: Univerzita Palackého v Olomouci, Katedra geoinformatiky
# Verze: 4.12
# Datum: 06.03.2025
#
# Popis:
# Tento skript provádí automatizovanou analýzu satelitních snímků Sentinel-2, včetně:
# - Ořezání rasterů pomocí masek
# - Výpočtu indexů NDVI a NDCI
# - Klasifikace indexů
# - Uložení výsledků do geodatabáze
# - Extrakce hodnot do bodových vrstev
# - Export do Excelu
#
# Použité knihovny:
# - arcpy
# - os
# - datetime
#
# Vstupní parametry:
# 1. root_folder: Cesta k adresáři obsahujícímu rastery
# 2. mask_fc_indices: Feature class pro ořezání indexových rasterů
# 3. mask_fc_rgb: Feature class pro ořezání RGB rasterů
# 4. points_fc: Bodová vrstva pro extrakci hodnot
# 5. output_name: Název výstupích souborů
# 6. workspace: Pracovní geodatabáze
# 7. excel_output: Cesta k Excel souboru
# 8. csv_output: Cesta k CSV souboru


import arcpy
import os
import datetime
from arcpy.sa import *

# Nastavení prostředí
arcpy.env.overwriteOutput = True
arcpy.env.parallelProcessingFactor = "100%"

def log_message(message):
    """Vypíše zprávu do ArcGIS a přidá časovou značku."""
    timestamp = datetime.datetime.now().strftime("%H:%M:%S")
    arcpy.AddMessage(f"[{timestamp}] {message}")

def find_raster(root_folder, band):
    """Najde raster odpovídající danému pásmu."""
    for dirpath, _, filenames in os.walk(root_folder):
        for filename in filenames:
            if f"_{band}_" in filename and filename.endswith(".jp2"):
                return os.path.join(dirpath, filename)
    raise FileNotFoundError(f"Rastr pro {band} nebyl nalezen.")

# Parametry
root_folder = arcpy.GetParameterAsText(0)
mask_fc_indices = arcpy.GetParameterAsText(1)
mask_fc_rgb = arcpy.GetParameterAsText(2)
points_fc = arcpy.GetParameterAsText(3)
output_name = arcpy.GetParameterAsText(4)
workspace = arcpy.GetParameterAsText(5)
excel_output = arcpy.GetParameterAsText(6)  # Cesta pro Excel

arcpy.env.workspace = workspace
log_message("Zpracování zahájeno...")

# Najdi rastery
raster_b8 = Raster(find_raster(os.path.join(root_folder, "R10m"), "B08"))
raster_b4 = Raster(find_raster(os.path.join(root_folder, "R10m"), "B04"))
raster_b5 = Raster(find_raster(os.path.join(root_folder, "R20m"), "B05"))
raster_b3 = Raster(find_raster(os.path.join(root_folder, "R10m"), "B03"))
raster_b2 = Raster(find_raster(os.path.join(root_folder, "R10m"), "B02"))

# Oříznutí pro indexy
log_message("Ořezávání rasterů pro indexy...")
raster_b8_clip = ExtractByMask(raster_b8, mask_fc_indices)
raster_b4_clip = ExtractByMask(raster_b4, mask_fc_indices)
raster_b5_clip = ExtractByMask(raster_b5, mask_fc_indices)

# Oříznutí pro RGB
log_message("Ořezávání rasterů pro RGB...")
raster_b4_rgb = ExtractByMask(raster_b4, mask_fc_rgb)
raster_b3_rgb = ExtractByMask(raster_b3, mask_fc_rgb)
raster_b2_rgb = ExtractByMask(raster_b2, mask_fc_rgb)

# Výpočet NDVI a NDCI
log_message("Výpočet NDVI a NDCI...")
ndvi = (raster_b8_clip - raster_b4_clip) / (raster_b8_clip + raster_b4_clip)
ndci = (raster_b5_clip - raster_b4_clip) / (raster_b5_clip + raster_b4_clip)

# Klasifikace NDVI a NDCI
log_message("Klasifikace NDVI a NDCI...")
ndvi_reclass = Reclassify(ndvi, "VALUE", "-1 -0.1 1; -0.1 0 2; 0 0.1 3; 0.1 0.2 4; 0.2 0.3 5; 0.3 1 6")
ndci_reclass = Reclassify(ndci, "VALUE", "-1 -0.1 1; -0.1 0 2; 0 0.1 3; 0.1 0.2 4; 0.2 0.3 5; 0.3 1 6")

# Uložení rasterů do GDB
log_message("Ukládání rasterů do geodatabáze...")
ndvi.save(os.path.join(workspace, f"{output_name}_NDVI"))
ndci.save(os.path.join(workspace, f"{output_name}_NDCI"))
ndvi_reclass.save(os.path.join(workspace, f"{output_name}_NDVI_klasifikovany"))
ndci_reclass.save(os.path.join(workspace, f"{output_name}_NDCI_klasifikovany"))

# Vytvoření RGB rastru
log_message("Vytváření RGB kompozitu...")
rgb_output = os.path.join(workspace, f"{output_name}_RGB")
arcpy.management.CompositeBands([raster_b4_rgb, raster_b3_rgb, raster_b2_rgb], rgb_output)

# Extrakce bodových hodnot
log_message("Extrahování hodnot NDVI a NDCI do bodových vrstev...")
ndvi_points = os.path.join(workspace, f"{output_name}_body_NDVI")
ndci_points = os.path.join(workspace, f"{output_name}_body_NDCI")
arcpy.sa.ExtractValuesToPoints(points_fc, ndvi, ndvi_points, "NONE")
arcpy.sa.ExtractValuesToPoints(points_fc, ndci, ndci_points, "NONE")

# Extrakce tabulek
log_message("Extrahování tabulek...")
ndvi_table = os.path.join(workspace, f"{output_name}_tabulka_NDVI")
ndci_table = os.path.join(workspace, f"{output_name}_tabulka_NDCI")
arcpy.management.CopyRows(ndvi_points, ndvi_table)
arcpy.management.CopyRows(ndci_points, ndci_table)

# Spojení tabulek
log_message("Spojování tabulek...")
final_table = os.path.join(workspace, f"{output_name}_FIN_hodnoty")
arcpy.management.AddField(ndci_table, "NDCI_VALUE", "DOUBLE")
arcpy.management.CalculateField(ndci_table, "NDCI_VALUE", "!RASTERVALU!", "PYTHON3")
arcpy.management.JoinField(ndvi_table, "OBJECTID", ndci_table, "OBJECTID", ["NDCI_VALUE"])
arcpy.management.AddField(ndvi_table, "NDVI_VALUE", "DOUBLE")
arcpy.management.CalculateField(ndvi_table, "NDVI_VALUE", "!RASTERVALU!", "PYTHON3")
arcpy.management.CopyRows(ndvi_table, final_table)

# Export do Excelu
if excel_output:
    log_message("Export do Excelu...")
    arcpy.conversion.TableToExcel(final_table, excel_output)

log_message("Skript úspěšně dokončen!")
